<?php
/**
 * Plugin Name: GardeVault Core
 * Description: Core services and console for GV plugins. Registry, audit, headers, Cloudflare, health, WP-CLI.
 * Version: 2.0.0
 * Author: GardeVault
 * Update URI: false
 * Text Domain: gardevault-core
 */

if (!defined('ABSPATH')) exit;

final class GV_Core {
    const VER     = '2.0.0';
    const OPT     = 'gv_options';
    const TB_AUD  = 'gv_audit';
    const MENU_SL = 'gardevault';

    private static $instance = null;
    public  $opts = [];
    private $registry = []; // slug => [name, version?, settings_url, panel_cb, cap]

    static function instance(){ return self::$instance ?: self::$instance = new self; }

    private function __construct(){
        $this->opts = wp_parse_args(get_option(self::OPT, []), [
            'cf_api_token'  => '',
            'cf_zone_id'    => '',
            'headers'       => [
                'enable'   => false,
                'hsts'     => false,
                'referrer' => 'strict-origin-when-cross-origin',
            ],
            'notifications' => [
                'email' => get_option('admin_email'),
                'level' => 'critical', // 'none', 'critical', 'all'
            ],
            'telemetry'     => false,
        ]);

        add_action('plugins_loaded', [$this, 'load_textdomain']);
        add_action('admin_init', [$this,'collect_registry'], 1);

        add_action('admin_menu', [$this, 'admin_menu']);
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), [$this, 'settings_link']);
        add_action('admin_enqueue_scripts', [$this, 'admin_assets']);
        add_action('admin_head', [$this, 'menu_icon_css']);

        add_action('send_headers', [$this, 'secure_headers'], 20);
        add_filter('site_status_tests', [$this,'health_tests']);

        add_action('gv_core_purge_urls', [$this, 'cf_purge_urls'], 10, 1);
        add_action('gv_core_purge_tags', [$this, 'cf_purge_tags'], 10, 1);
        add_action('gv_core_audit',      [$this, 'audit'],        10, 3);

        // Activation / Deactivation hooks
        if (function_exists('register_activation_hook')) {
            register_activation_hook(__FILE__, [__CLASS__,'activate']);
        }
        if (function_exists('register_deactivation_hook')) {
            register_deactivation_hook(__FILE__, [__CLASS__, 'deactivate']);
        }

        // Daily maintenance cron
        add_action('gv_core_daily_prune', [$this, 'run_daily_prune']);

        // Tool handlers
        add_action('admin_post_gv_core_tool_cf_purge_all',  [$this, 'handle_tool_cf_purge_all']);
        add_action('admin_post_gv_core_tool_cf_purge_home', [$this, 'handle_tool_cf_purge_home']);
        add_action('admin_post_gv_core_tool_prune_audit',   [$this, 'handle_tool_prune_audit']);


        add_action('rest_api_init', function () {
            register_rest_route('gv-core/v1', '/status', [
                'methods'=>'GET', 'permission_callback'=> function() {
                    return current_user_can('manage_options');
                },
                'callback'=>function(){ return ['version'=>self::VER,'cf'=>$this->is_cf_configured()?'configured':'not_configured','modules'=>array_values($this->registry)]; }
            ]);
        });

        if (defined('WP_CLI') && WP_CLI) WP_CLI::add_command('gv', [$this,'cli_root']);
    }

    function load_textdomain() {
        load_plugin_textdomain('gardevault-core', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    /* ---------- Install / Uninstall ---------- */
    static function activate(){
        global $wpdb; $charset=$wpdb->get_charset_collate(); $t=$wpdb->prefix.self::TB_AUD;
        require_once ABSPATH.'wp-admin/includes/upgrade.php';
        dbDelta("CREATE TABLE IF NOT EXISTS $t(
          id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
          ts DATETIME NOT NULL,
          actor VARCHAR(60) NOT NULL,
          action VARCHAR(120) NOT NULL,
          details LONGTEXT NULL,
          ctx LONGTEXT NULL,
          ip VARBINARY(16) NULL,
          ua TEXT NULL,
          PRIMARY KEY(id), KEY ts(ts), KEY action(action)
        ) $charset;");
        if (!get_option(self::OPT)) add_option(self::OPT, []);

        // Schedule daily maintenance
        if (!wp_next_scheduled('gv_core_daily_prune')) {
            wp_schedule_event(time(), 'daily', 'gv_core_daily_prune');
        }
    }

    static function deactivate(){
        wp_clear_scheduled_hook('gv_core_daily_prune');
    }

    /* ---------- Registry ---------- */
function collect_registry(){
    do_action('gv_core_register_module', $this);
    $this->fallback_register('gvcore','GV Core','admin.php?page='.self::MENU_SL);
    $this->fallback_register('gvforms','GV Contact Form Pro','admin.php?page=gvforms');
    $this->fallback_register('gv2fa','GV Simple 2FA','options-general.php?page=gv2fa_settings');
    $this->fallback_register('gvfw','GV Simple Firewall','options-general.php?page=gvfw_settings');
}
    private function fallback_register($slug,$name,$path){
        if (isset($this->registry[$slug])) return;
        $this->registry[$slug] = [
            'slug'=>$slug,'name'=>$name,'settings_url'=>admin_url($path),
            'panel_cb'=>null,'cap'=>'manage_options','version'=>''
        ];
    }
    public function register_module($args){
        $d = wp_parse_args($args,[
            'slug'=>'','name'=>'','settings_url'=>'','panel_cb'=>null,'cap'=>'manage_options','version'=>''
        ]);
        if (!$d['slug'] || !$d['name']) return;
        $this->registry[$d['slug']] = $d;
    }

    /* ---------- Admin UI ---------- */
    function admin_menu(){
        $icon = plugin_dir_url(__FILE__).'assets/imgs/gardevault-logo-mini.svg';
        add_menu_page(
            __('GardeVault Console', 'gardevault-core'),
            'GardeVault',
            'manage_options',
            self::MENU_SL,
            [$this,'render_console'],
            $icon,
            60
        );
    }
    function settings_link($links){
        $links[]='<a href="'.admin_url('admin.php?page='.self::MENU_SL).'">'.esc_html__('Console', 'gardevault-core').'</a>'; return $links;
    }

    function admin_assets($hook){
        if (strpos($hook, self::MENU_SL) === false) return;

        // Enqueue admin notices
        wp_enqueue_style('wp-components');
        wp_enqueue_script('wp-notices');

        wp_register_style('gv-core-admin', false);
        wp_enqueue_style('gv-core-admin');

        wp_add_inline_style('gv-core-admin', '
          /* ---------- Design tokens ---------- */
          .gv-wrap{
            --gv-radius: 10px;
            --gv-gap: 12px;
            --gv-font: system-ui, -apple-system, Segoe UI, Roboto, Helvetica, Arial, sans-serif;
            --gv-surface: #ffffff;
            --gv-elev: #f6f7f7;
            --gv-border: #dcdcde;
            --gv-muted: #646970;
            --gv-text: #1c95e6ff;
            --gv-accent: #2271b1;
            --gv-accent-weak: #72aee6;
            max-width:1200px;
            font-family: var(--gv-font);
            color: var(--gv-text);
          }

          /* Header */
          .gv-head{display:flex;justify-content:space-between;align-items:center;margin:14px 0 16px}
          .gv-badge{font-family:ui-monospace, SFMono-Regular, Menlo, Consolas, monospace;
            padding:2px 8px;border-radius:6px;background:var(--gv-elev);border:1px solid var(--gv-border)}

          /* Tabs – card style */
          .gv-tabs{display:flex;gap:8px;margin:0 0 14px;border-bottom:1px solid var(--gv-border)}
          .gv-tabs a{
            display:inline-block;padding:8px 12px;border:1px solid var(--gv-border);
            border-bottom:none;border-top-left-radius:8px;border-top-right-radius:8px;
            background:var(--gv-surface);text-decoration:none;color:inherit;
            transition:background .15s ease,border-color .15s ease, box-shadow .15s ease;
          }
          .gv-tabs a:hover{background:var(--gv-elev)}
          .gv-tabs a.active{
            font-weight:600;
            box-shadow: 0 1px 0 0 var(--gv-surface);
          }

          /* Panels */
          .gv-panel{display:none;background:var(--gv-surface);padding:14px;border:1px solid var(--gv-border);
            border-radius:0 0 var(--gv-radius) var(--gv-radius)}
          .gv-panel.active{display:block}

          /* Table */
          .gv-table{width:100%;border-collapse:separate;border-spacing:0;background:var(--gv-surface);
            border:1px solid var(--gv-border);border-radius:var(--gv-radius);overflow:hidden}
          .gv-table th,.gv-table td{padding:10px 12px;border-bottom:1px solid var(--gv-border);text-align:left}
          .gv-table thead th{background:var(--gv-elev);font-weight:600}
          .gv-table tbody tr:hover{background:rgba(34,113,177,.06)}
          .gv-table tbody tr:last-child td{border-bottom:none}
          .form-table th, .form-wrap label {color: white}
     h2 {color: white}
          /* Status */
          .gv-status{font-weight:600}
          .gv-ok{color:#1a7f37}
          .gv-bad{color:#d63638}

          /* Help text and callouts */
          .gv-help{color:var(--gv-muted);margin:.35rem 0 0}
          .gv-callout{margin:14px 0;padding:12px;border:1px solid var(--gv-border);background:var(--gv-elev);
            border-left:4px solid var(--gv-accent-weak);border-radius:8px}
          .gv-kbd{font-family:ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, monospace;
            padding:.15rem .35rem;border-radius:4px;background:#eef1f4;border:1px solid var(--gv-border)}

          /* Details toggle */
          details.gv-more summary{cursor:pointer;font-weight:600}
          details.gv-more{margin-top:8px}

          /* Focus ring to match WP */
          .gv-tabs a:focus,
          .gv-panel a:focus,
          .gv-panel input:focus,
          .gv-panel select:focus,
          .gv-panel textarea:focus{
            outline:2px solid transparent;
            box-shadow:0 0 0 1px var(--gv-accent),0 0 0 3px rgba(34,113,177,.25);
            border-color:var(--gv-accent);
          }

          /* Header actions */
    .gv-head{gap:12px}
    .gv-head .gv-actions{display:flex;gap:8px;align-items:center}
    .gv-head .button.button-primary{
      background:var(--gv-accent);border-color:var(--gv-accent);
      box-shadow:none
    }
    .gv-head .button.button-primary:hover{filter:brightness(1.07)}
    /* Tighten tables */
    .gv-table th,.gv-table td{padding:9px 12px}

          /* Dark mode */
          @media (prefers-color-scheme: dark){
            .gv-wrap{
              --gv-surface:#1e1f20;
              --gv-elev:#242628;
              --gv-border:#2b2d30;
              --gv-muted:#9aa0a6;
              --gv-text:#e6e6e6;
              --gv-accent:#5ba8ff;
              --gv-accent-weak:#5ba8ff;
            }
            .gv-kbd{background:#2b2d30}
            .gv-tabs a.active{box-shadow: 0 1px 0 0 var(--gv-surface)}
            .gv-table tbody tr:hover{background:rgba(91,168,255,.08)}
          }

          .gv-table .button{
      font-size:13px;
      line-height:1.5;
      padding:2px 8px;
      border-radius:4px;
    background: #2a7b9b62;
    background: radial-gradient(circle,rgba(42, 123, 155, 0.35) 0%, rgba(27, 14, 82, 0.27) 100%);
      color: white;
    }
    .gv-table .button:hover{
      filter:brightness(1.1);
    }

          /* Motion respect */
          @media (prefers-reduced-motion: reduce){
            .gv-tabs a{transition:none}
          }
        ');

        wp_register_script('gv-core-admin-js', false);
        wp_enqueue_script('gv-core-admin-js');
        wp_add_inline_script('gv-core-admin-js',
          "document.addEventListener('click',function(e){
             var t=e.target.closest('[data-gv-tab]'); if(!t) return;
             e.preventDefault();
             document.querySelectorAll('[data-gv-tab]').forEach(a=>a.classList.remove('active'));
             document.querySelectorAll('.gv-panel').forEach(p=>p.classList.remove('active'));
             t.classList.add('active');
             var id=t.getAttribute('data-gv-tab'); var el=document.getElementById(id);
             if(el) el.classList.add('active');
             // Persist tab
             if(history.pushState) history.pushState(null,null,'#gv-tab='+id);
           });
           // Restore tab on load
           window.addEventListener('load', function(){
             var hash = window.location.hash.match(/#gv-tab=([^&]+)/);
             if(!hash) return;
             var tab = document.querySelector('[data-gv-tab=\"'+hash[1]+'\"]');
             if(tab) tab.click();
           });
          "
        );
    }


    public function menu_icon_css(){
        echo '<style>
          #toplevel_page_gardevault .wp-menu-image img,
          #toplevel_page_gardevault .wp-menu-image svg{
            width:20px;height:20px;display:block;object-fit:contain;justify-content:center;align-items: anchor-center; justify-self: center;
          }
          #toplevel_page_gardevault .wp-menu-image img{opacity:.75}
          #toplevel_page_gardevault.wp-has-current-submenu .wp-menu-image img,
          #toplevel_page_gardevault.current .wp-menu-image img{opacity:1}
        </style>';
    }

    function render_console(){
        if (!current_user_can('manage_options')) return;

        // Save
        if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['gv_core_save']) && check_admin_referer('gv_save')) {
            $this->opts['cf_api_token']               = sanitize_text_field($_POST['cf_api_token'] ?? '');
            $this->opts['cf_zone_id']                 = sanitize_text_field($_POST['cf_zone_id'] ?? '');
            $this->opts['headers']['enable']          = !empty($_POST['headers_enable']);
            $this->opts['headers']['hsts']            = !empty($_POST['headers_hsts']);
            $this->opts['headers']['referrer']        = sanitize_text_field($_POST['headers_referrer'] ?? 'strict-origin-when-cross-origin');
            $this->opts['notifications']['email']     = sanitize_email($_POST['notify_email'] ?? '');
            $this->opts['notifications']['level']     = sanitize_key($_POST['notify_level'] ?? 'critical');
            
            update_option(self::OPT, $this->opts);
            add_settings_error('gv-core', 'settings_saved', __('Saved.', 'gardevault-core'), 'updated');
            gv_audit('core_settings_saved', ['user'=>get_current_user_id()]);
        }

        // Show notices
        settings_errors('gv-core');

        // Modules → fill versions/active
        require_once ABSPATH.'wp-admin/includes/plugin.php';
        foreach (get_plugins() as $file=>$data){
            $is_gv = (stripos($data['Name'],'GV ')===0 || stripos($data['Author'],'Gardevault')!==false);
            if (!$is_gv) continue;
            foreach ($this->registry as $k=>$m) {
                if ($m['name'] === $data['Name'] || stripos($data['Name'],$m['slug'])!==false) {
                    $this->registry[$k]['version'] = $data['Version'];
                    $this->registry[$k]['active']  = is_plugin_active($file);
                }
            }
        }

$core_file = plugin_basename(__FILE__);
$this->registry['gvcore']['version'] = self::VER;
$this->registry['gvcore']['active']  = is_plugin_active($core_file) || true; // running plugin

        echo '<div class="wrap gv-wrap">';
        echo '<div class="gv-head"><h1>'.esc_html__('GardeVault Console', 'gardevault-core').'</h1><span class="gv-badge">Core '.esc_html(self::VER).'</span></div>';
        echo '<div class="gv-tabs">';
        echo '<a href="#" class="active" data-gv-tab="gv-modules">'.esc_html__('Modules', 'gardevault-core').'</a>';
        echo '<a href="#" data-gv-tab="gv-audit">'.esc_html__('Audit', 'gardevault-core').'</a>';
        echo '<a href="#" data-gv-tab="gv-tools">'.esc_html__('Tools', 'gardevault-core').'</a>';
        echo '<a href="#" data-gv-tab="gv-settings">'.esc_html__('Settings', 'gardevault-core').'</a>';
        echo '</div>';

        /* -------- Modules -------- */
        echo '<div id="gv-modules" class="gv-panel active">';
        echo '<div class="gv-callout">'.sprintf(
            esc_html__('Download or update more Gardevault plugins below!
             Use %s to open each plugin\'s own screen. The Quick Panel shows common actions provided by that plugin.', 'gardevault-core'),
            '<span class="gv-kbd">'.esc_html__('Settings', 'gardevault-core').'</span>'
        ).'</div>';
       echo '<table class="gv-table">
      <thead>
        <tr>
          <th>'.esc_html__('Plugin', 'gardevault-core').'</th>
          <th>'.esc_html__('Version', 'gardevault-core').'</th>
          <th>'.esc_html__('Status', 'gardevault-core').'</th>
          <th>'.esc_html__('Settings', 'gardevault-core').'</th>
          <th>'.esc_html__('Quick Panel', 'gardevault-core').'</th>
          <th>'.esc_html__('Download / Update', 'gardevault-core').'</th>
        </tr>
      </thead>
      <tbody>';
      // before foreach ($this->registry as $m) { ... }
$dl_map = [
  'gvfw'    => 'https://staging.gardevault.eu/wp-content/uploads/2025/10/gv-simple-firewall.zip',
  'gv2fa'   => 'https://staging.gardevault.eu/wp-content/uploads/2025/10/wp-2fa.zip',
  'gvcore'  => 'https://staging.gardevault.eu/wp-content/uploads/2025/10/gv-core.zip',
  'gvforms' => 'https://staging.gardevault.eu/wp-content/uploads/2025/10/Gv-Forms-booking.zip',
];
$gh_map = [
  'gvfw'    => 'https://github.com/Davekrush/GV-Simple-Firewall',
  'gv2fa'   => 'https://github.com/Davekrush/GV-2FA',
  'gvcore'  => 'https://github.com/Davekrush/GV-Core-plugin',
  'gvforms' => 'https://github.com/Davekrush/GV-Contact-Form-Pro',
];

foreach ($this->registry as $m){
    $status = !empty($m['active'])
        ? '<span class="gv-status gv-ok">'.esc_html__('Active','gardevault-core').'</span>'
        : '<span class="gv-status gv-bad">'.esc_html__('Inactive','gardevault-core').'</span>';
    $settingsLink = $m['settings_url'] ? '<a href="'.esc_url($m['settings_url']).'">'.esc_html__('Open','gardevault-core').'</a>' : '&mdash;';
    $panelHtml = '&mdash;';
    if (is_callable($m['panel_cb']) && current_user_can($m['cap'] ?? 'manage_options')) {
        ob_start(); call_user_func($m['panel_cb']); $panelHtml = ob_get_clean();
    }

    $dl  = $dl_map[$m['slug']]  ?? '';
    $gh  = $gh_map[$m['slug']]  ?? '';
    $btns = $dl ? '<a class="button" target="_blank" rel="noopener" href="'.esc_url($dl).'">'.esc_html__('Download','gardevault-core').'</a>' : '&mdash;';
    if ($gh) $btns .= ' <a class="button" target="_blank" rel="noopener" href="'.esc_url($gh).'">'.esc_html__('GitHub','gardevault-core').'</a>';

    echo '<tr>
            <td>'.esc_html($m['name']).'</td>
            <td>'.esc_html($m['version']).'</td>
            <td>'.$status.'</td>
            <td>'.$settingsLink.'</td>
            <td>'.$panelHtml.'</td>
            <td>'.$btns.'</td>
          </tr>';
}

        echo '</tbody></table></div>';

        /* -------- Audit -------- */
        echo '<div id="gv-audit" class="gv-panel">';
        echo '<div class="gv-callout">'.sprintf(
            esc_html__('Recent core events. Use WP-CLI to view more: %s.', 'gardevault-core'),
            '<code class="gv-kbd">wp gv log --limit=200</code>'
        ).'</div>';
        global $wpdb; $t=$wpdb->prefix.self::TB_AUD;
        $rows = $wpdb->get_results("SELECT * FROM $t ORDER BY id DESC LIMIT 200", ARRAY_A);
        echo '<table class="gv-table"><thead><tr>
            <th>'.esc_html__('Time', 'gardevault-core').'</th>
            <th>'.esc_html__('Actor', 'gardevault-core').'</th>
            <th>'.esc_html__('Action', 'gardevault-core').'</th>
            <th>'.esc_html__('Details', 'gardevault-core').'</th>
            </tr></thead><tbody>';
        foreach ($rows as $r){
            echo '<tr><td>'.esc_html($r['ts']).'</td><td>'.esc_html($r['actor']).'</td><td>'.esc_html($r['action']).'</td><td><code>'.esc_html(mb_strimwidth((string)$r['details'],0,400,'…')).'</code></td></tr>';
        }
        if (!$rows) echo '<tr><td colspan="4">'.esc_html__('No entries yet.', 'gardevault-core').'</td></tr>';
        echo '</tbody></table></div>';

        /* -------- Tools -------- */
        echo '<div id="gv-tools" class="gv-panel">';
        
        echo '<h2>'.esc_html__('Cache Tools', 'gardevault-core').'</h2>';
        echo '<p class="gv-help">'.esc_html__('Use these tools to clear Cloudflare caches. (Requires Cloudflare to be configured in Settings).', 'gardevault-core').'</p>';
        echo '<table class="form-table">
                <tr>
                    <th>'.esc_html__('Purge Homepage', 'gardevault-core').'</th>
                    <td>
                        <a href="'.wp_nonce_url(admin_url('admin-post.php?action=gv_core_tool_cf_purge_home'), 'gv_tool_cf_purge_home').'" class="button">'.esc_html__('Purge Homepage', 'gardevault-core').'</a>
                        <p class="description">'.esc_html__('Clears the cache for your site\'s front page.', 'gardevault-core').'</p>
                    </td>
                </tr>
                <tr>
                    <th>'.esc_html__('Purge Everything', 'gardevault-core').'</th>
                    <td>
                        <a href="'.wp_nonce_url(admin_url('admin-post.php?action=gv_core_tool_cf_purge_all'), 'gv_tool_cf_purge_all').'" class="button button-primary">'.esc_html__('Purge Entire Cache', 'gardevault-core').'</a>
                        <p class="description">'.esc_html__('Warning: This will clear all cached files on Cloudflare and may temporarily slow down your site.', 'gardevault-core').'</p>
                    </td>
                </tr>
            </table>';

        echo '<h2>'.esc_html__('Maintenance Tools', 'gardevault-core').'</h2>';
        echo '<table class="form-table">
                <tr>
                    <th>'.esc_html__('Prune Audit Log', 'gardevault-core').'</th>
                    <td>
                        <a href="'.wp_nonce_url(admin_url('admin-post.php?action=gv_core_tool_prune_audit'), 'gv_tool_prune_audit').'" class="button">'.esc_html__('Prune Logs Now', 'gardevault-core').'</a>
                        <p class="description">'.esc_html__('Deletes all audit log entries older than 90 days. This runs automatically once per day.', 'gardevault-core').'</p>
                    </td>
                </tr>
            </table>';

        echo '</div>';

        /* -------- Settings -------- */
        $o=$this->opts;
        echo '<div id="gv-settings" class="gv-panel"><form method="post">';
        wp_nonce_field('gv_save'); echo '<input type="hidden" name="gv_core_save" value="1" />';

        echo '<h2>'.esc_html__('Cloudflare', 'gardevault-core').'</h2>
          <p class="gv-help">'.esc_html__('Optional. If set, GV can purge CDN cache for specific URLs or tags from plugins that support it.', 'gardevault-core').'</p>
          <table class="form-table">
            <tr>
              <th>'.esc_html__('API Token', 'gardevault-core').'</th>
              <td>
                <input type="password" name="cf_api_token" value="'.esc_attr($o['cf_api_token']).'" class="regular-text" />
                <p class="description">'.sprintf(
                    esc_html__('Create a %s in Cloudflare with the permission %s. Keep it secret.', 'gardevault-core'),
                    '<strong>'.esc_html__('Scoped API Token', 'gardevault-core').'</strong>',
                    '<code>Zone &rarr; Cache Purge</code>'
                ).'</p>
              </td>
            </tr>
            <tr>
              <th>'.esc_html__('Zone ID', 'gardevault-core').'</th>
              <td>
                <input name="cf_zone_id" value="'.esc_attr($o['cf_zone_id']).'" class="regular-text" />
                <p class="description">'.sprintf(
                    esc_html__('Your site\'s Cloudflare %s. Find it on your Cloudflare dashboard &rarr; Overview.', 'gardevault-core'),
                    '<em>'.esc_html__('Zone ID', 'gardevault-core').'</em>'
                ).'</p>
              </td>
            </tr>
            <tr>
              <th>'.esc_html__('Status', 'gardevault-core').'</th>
              <td>'.($this->is_cf_configured()
                    ? '<span class="gv-ok">'.esc_html__('Configured', 'gardevault-core').'</span>'
                    : '<span class="gv-bad">'.esc_html__('Not configured', 'gardevault-core').'</span>'
              ).'
                <p class="gv-help">'.sprintf(
                    esc_html__('When configured, you may use WP-CLI to purge: %s or %s.', 'gardevault-core'),
                    '<code class="gv-kbd">wp gv purge --url=https://example.com/page/</code>',
                    '<code class="gv-kbd">--tag=products</code>'
                ).'</p>
              </td>
            </tr>
          </table>';

        echo '<h2>'.esc_html__('Security Headers', 'gardevault-core').'</h2>
          <p class="gv-help">'.esc_html__('Adds light-touch headers at PHP level. If your server or CDN already sets these, leave disabled to avoid duplicates.', 'gardevault-core').'</p>
          <table class="form-table">
            <tr>
              <th scope="row">'.esc_html__('Baseline', 'gardevault-core').'</th>
              <td>
                <label><input type="checkbox" name="headers_enable" '.checked(!empty($o['headers']['enable']),true,false).'/> '.esc_html__('Enable baseline headers', 'gardevault-core').'</label>
                <p class="description">'.sprintf(
                    esc_html__('Sets %s, %s, and disables legacy %s.', 'gardevault-core'),
                    '<code>X-Content-Type-Options: nosniff</code>', '<code>X-Frame-Options: SAMEORIGIN</code>', '<code>X-XSS-Protection</code>'
                ).'</p>
              </td>
            </tr>
            <tr>
              <th scope="row">'.esc_html__('HSTS', 'gardevault-core').'</th>
              <td>
                <label><input type="checkbox" name="headers_hsts" '.checked(!empty($o['headers']['hsts']),true,false).'/> '.esc_html__('Add HSTS (includeSubDomains; preload)', 'gardevault-core').'</label>
                <p class="description">'.sprintf(
                    esc_html__('Requires HTTPS. Sends %s with a 1-year max-age. Only enable after verifying all subdomains support HTTPS.', 'gardevault-core'),
                    '<code>Strict-Transport-Security</code>'
                ).'</p>
              </td>
            </tr>
            <tr>
              <th scope="row">'.esc_html__('Referrer-Policy', 'gardevault-core').'</th>
              <td>
                <input name="headers_referrer" value="'.esc_attr($o['headers']['referrer']).'" class="regular-text" />
                <p class="description">'.sprintf(
                    esc_html__('Controls what referrer data the browser sends. Default %s is safe for analytics while limiting cross-site leakage.', 'gardevault-core'),
                    '<code>strict-origin-when-cross-origin</code>'
                ).'</p>
              </td>
            </tr>
          </table>';

        echo '<h2>' . esc_html__('Notifications', 'gardevault-core') . '</h2>';
        echo '<p class="gv-help">' . esc_html__('Get email alerts for critical events from GardeVault plugins (e.g., firewall lockouts, 2FA changes).', 'gardevault-core') . '</p>';
        echo '<table class="form-table">';
        // Email field
        echo '<tr><th>' . esc_html__('Notification Email', 'gardevault-core') . '</th>';
        echo '<td><input type="email" name="notify_email" value="'.esc_attr($o['notifications']['email']).'" class="regular-text" placeholder="'.esc_attr(get_option('admin_email')).'" />';
        echo '<p class="description">' . esc_html__('Where to send alerts. Defaults to the site admin email if left empty.', 'gardevault-core') . '</p></td></tr>';
        // Level field
        $lvl = $o['notifications']['level'];
        echo '<tr><th>' . esc_html__('Notification Level', 'gardevault-core') . '</th><td>';
        echo '<select name="notify_level">';
        echo '<option value="none" '.selected($lvl, 'none', false).'>' . esc_html__('None', 'gardevault-core') . '</option>';
        echo '<option value="critical" '.selected($lvl, 'critical', false).'>' . esc_html__('Critical Only', 'gardevault-core') . '</option>';
        echo '<option value="all" '.selected($lvl, 'all', false).'>' . esc_html__('All Audit Events', 'gardevault-core') . '</option>';
        echo '</select><p class="description">' . esc_html__('"Critical Only" is recommended.', 'gardevault-core') . '</p></td></tr>';
        echo '</table>';


        echo '<details class="gv-more"><summary>'.esc_html__('Advanced notes', 'gardevault-core').'</summary>
          <ul style="margin:.5rem 0 0 1rem;list-style:disc">
            <li>'.esc_html__('Headers are only added if they are not already present. CDN/server values always win.', 'gardevault-core').'</li>
            <li>'.esc_html__('Cloudflare actions run server-side via the REST API. Network errors are logged to the audit table.', 'gardevault-core').'</li>
            <li>'.sprintf(
                esc_html__('All settings are stored in option %s. Export with %s.', 'gardevault-core'),
                '<code>gv_options</code>', '<span class="gv-kbd">wp option get gv_options</span>'
            ).'</li>
          </ul>
        </details>';

        submit_button(__('Save', 'gardevault-core'));
        echo '</form></div>';

        echo '</div>'; // .gv-wrap
    }

    /* ---------- Headers ---------- */
    function secure_headers(){
        if (headers_sent() || empty($this->opts['headers']['enable'])) return;
        header('X-Content-Type-Options: nosniff', false);
        header('X-Frame-Options: SAMEORIGIN', false);
        header('X-XSS-Protection: 0', false);
        if (!$this->header_present('Referrer-Policy')) header('Referrer-Policy: '.$this->opts['headers']['referrer'], false);
        if (!empty($this->opts['headers']['hsts']) && is_ssl()) header('Strict-Transport-Security: max-age=31536000; includeSubDomains; preload', false);
    }
    private function header_present($name){ foreach (headers_list() as $h){ if (stripos($h,$name.':')===0) return true; } return false; }

    /* ---------- Cloudflare ---------- */
    private function is_cf_configured(){ return !empty($this->opts['cf_api_token']) && !empty($this->opts['cf_zone_id']); }
    
    function cf_purge_urls($urls=[]){
        $urls=array_values(array_filter(array_map('esc_url_raw',(array)$urls)));
        if(!$urls) return new WP_Error('gv_cf_nothing','No URLs to purge.');
        return $this->cf_call('purge_cache',['files'=>$urls]);
    }
    function cf_purge_tags($tags=[]){
        $tags=array_values(array_filter(array_map('sanitize_key',(array)$tags)));
        if(!$tags) return new WP_Error('gv_cf_nothing','No tags to purge.');
        return $this->cf_call('purge_cache',['tags'=>$tags]);
    }
    function cf_purge_all(){
        return $this->cf_call('purge_cache', ['purge_everything' => true]);
    }

    private function cf_call($path,$payload){
        if(!$this->is_cf_configured()) {
            $error = new WP_Error('gv_cf_config','Cloudflare not configured.');
            gv_audit('cf_call_failed', $error->get_error_message());
            return $error;
        }
        $r=wp_remote_post("https://api.cloudflare.com/client/v4/zones/{$this->opts['cf_zone_id']}/$path",[
            'headers'=>['Authorization'=>'Bearer '.$this->opts['cf_api_token'],'Content-Type'=>'application/json'],
            'timeout'=>12,'body'=>wp_json_encode($payload),
        ]);
        if(is_wp_error($r)) {
            gv_audit('cf_call_failed', $r->get_error_message());
            return $r;
        }
        $code=wp_remote_retrieve_response_code($r);
        if($code>=200 && $code<300) {
            gv_audit('cf_call_success', $path, $payload);
            return true;
        }
        
        $error = new WP_Error('gv_cf_http',"HTTP $code: ".wp_remote_retrieve_body($r));
        gv_audit('cf_call_failed', $error->get_error_message());
        return $error;
    }

    /* ---------- Audit & Maintenance ---------- */
    function audit($action,$details='',$ctx=[]){
        global $wpdb; $t=$wpdb->prefix.self::TB_AUD;
        $actor = is_user_logged_in() ? wp_get_current_user()->user_login : 'system';
        $ip = isset($_SERVER['REMOTE_ADDR']) ? @inet_pton($_SERVER['REMOTE_ADDR']) : null;
        $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $wpdb->insert($t,[
            'ts'=>current_time('mysql'),
            'actor'=>substr((string)$actor,0,60),
            'action'=>substr(sanitize_text_field($action),0,120),
            'details'=>is_string($details)?$details:wp_json_encode($details),
            'ctx'=>$ctx?wp_json_encode($ctx):null,
            'ip'=>$ip,'ua'=>$ua
        ]);
    }

    function run_daily_prune(){
        global $wpdb;
        $table = $wpdb->prefix . self::TB_AUD;
        $retention_days = apply_filters('gv_core_log_retention_days', 90);
        
        $result = $wpdb->query( $wpdb->prepare(
            "DELETE FROM $table WHERE ts < %s",
            date('Y-m-d H:i:s', strtotime("-$retention_days days"))
        ) );

        if ($result !== false) {
            gv_audit('audit_prune_success', "Pruned $result rows older than $retention_days days.");
        }
    }

    /* ---------- Health ---------- */
    function health_tests($tests){
        $tests['direct']['gv_cf']=['label'=>__('GardeVault: Cloudflare purge', 'gardevault-core'),'test'=>function(){
            $ok=$this->is_cf_configured();
            return [
                'label'       => $ok ? __('Cloudflare configured', 'gardevault-core') : __('Cloudflare not configured', 'gardevault-core'),
                'status'      => $ok ? 'good' : 'recommended',
                'description' => $ok 
                    ? __('API token and zone ID set.', 'gardevault-core') 
                    : __('Set API token and zone ID in GardeVault → Console.', 'gardevault-core')
            ];
        }];
        return $tests;
    }

    /* ---------- Admin Tool Handlers ---------- */
    
    /** Helper to redirect back to our console with a message */
    private function tool_redirect($type = 'success', $message = '') {
        if ($message) {
            add_settings_error('gv-core', 'tool_action', $message, $type);
        }
        $url = admin_url('admin.php?page=' . self::MENU_SL . '&gv-tab=gv-tools');
        wp_safe_redirect(add_query_arg('settings-updated', 'true', $url));
        exit;
    }

    function handle_tool_cf_purge_all() {
        if (!current_user_can('manage_options')) wp_die(__('Permission denied', 'gardevault-core'));
        check_admin_referer('gv_tool_cf_purge_all');

        $result = $this->cf_purge_all();
        if (is_wp_error($result)) {
            $this->tool_redirect('error', __('Cloudflare Purge All Failed:', 'gardevault-core') . ' ' . $result->get_error_message());
        } else {
            $this->tool_redirect('success', __('Cloudflare "Purge Everything" request sent successfully.', 'gardevault-core'));
        }
    }

    function handle_tool_cf_purge_home() {
        if (!current_user_can('manage_options')) wp_die(__('Permission denied', 'gardevault-core'));
        check_admin_referer('gv_tool_cf_purge_home');

        $result = $this->cf_purge_urls([home_url('/')]);
        if (is_wp_error($result)) {
            $this->tool_redirect('error', __('Cloudflare Homepage Purge Failed:', 'gardevault-core') . ' ' . $result->get_error_message());
        } else {
            $this->tool_redirect('success', __('Cloudflare homepage purge request sent successfully.', 'gardevault-core'));
        }
    }

    function handle_tool_prune_audit() {
        if (!current_user_can('manage_options')) wp_die(__('Permission denied', 'gardevault-core'));
        check_admin_referer('gv_tool_prune_audit');
        
        // Use the same logic as the cron
        $this->run_daily_prune();
        
        $this->tool_redirect('success', __('Audit log pruned successfully.', 'gardevault-core'));
    }


    /* ---------- WP-CLI ---------- */
    function cli_root($args,$assoc){
        $sub=$args[0]??'';
        switch($sub){
            case 'log': $this->cli_log($assoc); break;
            case 'purge':
                $urls=$assoc['url']??null; $tags=$assoc['tag']??null; $all=$assoc['all']??null;
                if ($all) {
                    $r = $this->cf_purge_all();
                } elseif ($urls) {
                    $r = $this->cf_purge_urls((array)$urls);
                } elseif ($tags) {
                    $r = $this->cf_purge_tags((array)$tags);
                } else {
                    $r = new WP_Error('args','Usage: --url=<url>, --tag=<tag>, or --all');
                }
                is_wp_error($r)?WP_CLI::error($r->get_error_message()):WP_CLI::success('Purge requested.'); break;
            case 'prune':
                $this->run_daily_prune();
                WP_CLI::success('Audit log pruned.'); break;
            default: WP_CLI::line("Usage:\n  wp gv log [--limit=50]\n  wp gv purge --url=<url>\n  wp gv purge --tag=<tag>\n  wp gv purge --all\n  wp gv prune");
        }
    }
    private function cli_log($assoc){
        global $wpdb; $limit=isset($assoc['limit'])?max(1,intval($assoc['limit'])):50; $t=$wpdb->prefix.self::TB_AUD;
        $rows=$wpdb->get_results($wpdb->prepare("SELECT * FROM $t ORDER BY id DESC LIMIT %d",$limit),ARRAY_A);
        if (empty($rows)) {
            WP_CLI::line('No audit logs found.'); return;
        }
        foreach($rows as $r){ WP_CLI::line("{$r['id']} {$r['ts']} {$r['actor']} {$r['action']} ".mb_strimwidth((string)$r['details'],0,120,'…')); }
    }
}

GV_Core::instance();

/* ---------- Stable helpers ---------- */
function gv_core_ready($min='2.0.0'){ return class_exists('GV_Core') && version_compare(GV_Core::VER,$min,'>='); }
function gv_audit($action,$details='',$ctx=[]){ if(gv_core_ready()) do_action('gv_core_audit',$action,$details,$ctx); }
function gv_cf_purge_urls($urls){ if(gv_core_ready()) do_action('gv_core_purge_urls',(array)$urls); }
function gv_cf_purge_tags($tags){ if(gv_core_ready()) do_action('gv_core_purge_tags',(array)$tags); }

/* ---------- Registry helper for plugins ---------- */
function gv_core_register_module($args){
    if (!gv_core_ready()) return;
    GV_Core::instance()->register_module($args);
}

/* ---------- Notification helper ---------- */
function gv_notify($action, $details = '', $level = 'info') {
    // 1. Always log the event
    gv_audit($action, $details);

    // 2. Check if we should email
    $core = GV_Core::instance();
    if (empty($core->opts['notifications']['email'])) return;

    $notify_level = $core->opts['notifications']['level'] ?? 'critical';
    if ($notify_level === 'none') return;

    $should_notify = ($level === 'critical' && $notify_level === 'critical') || 
                     ($notify_level === 'all');

    if ($should_notify) {
        $subject = "[GardeVault Event] - " . $action;
        $body = "A '$level' event occurred on your site:\n\n";
        $body .= "Action: $action\n";
        $body .= "Details: " . (is_string($details) ? $details : wp_json_encode($details, JSON_PRETTY_PRINT));
        wp_mail($core->opts['notifications']['email'], $subject, $body);
    }
}